<?php
// Robust HLS proxy to allow playing HTTP streams from HTTPS pages.
// - Fetches playlists and segments via cURL
// - Rewrites .m3u8 so that ALL URIs (variants + segments) point back to this proxy
// - Supports Range requests for segments
// - Sets correct Content-Type headers

header("Cache-Control: no-store, no-cache, must-revalidate, max-age=0");
header("Pragma: no-cache");

if (!isset($_GET['u'])) { http_response_code(400); echo "missing u"; exit; }
$u = base64_decode($_GET['u'], true);
if (!$u) { http_response_code(400); echo "bad base64"; exit; }

$parts = parse_url($u);
if (!$parts || !isset($parts['scheme']) || !isset($parts['host'])) { http_response_code(400); echo "bad url"; exit; }
$scheme = strtolower($parts['scheme']);
if ($scheme !== 'http' && $scheme !== 'https') { http_response_code(400); echo "unsupported scheme"; exit; }

// Build headers to send upstream
$headers = [
  "User-Agent: TV360-IPTV/1.0 (+proxy)",
  "Accept: */*",
  "Connection: close"
];
if (!empty($_SERVER['HTTP_RANGE'])) {
  $headers[] = "Range: " . $_SERVER['HTTP_RANGE'];
}

// cURL fetch helper
function curl_fetch($url, $headers){
  $ch = curl_init();
  curl_setopt_array($ch, [
    CURLOPT_URL => $url,
    CURLOPT_RETURNTRANSFER => true,
    CURLOPT_FOLLOWLOCATION => true,
    CURLOPT_CONNECTTIMEOUT => 10,
    CURLOPT_TIMEOUT => 30,
    CURLOPT_HTTPHEADER => $headers,
    CURLOPT_HEADER => true,
  ]);
  $resp = curl_exec($ch);
  if ($resp === false) {
    $err = curl_error($ch);
    curl_close($ch);
    return [null, null, 0, $err];
  }
  $header_size = curl_getinfo($ch, CURLINFO_HEADER_SIZE);
  $status = curl_getinfo($ch, CURLINFO_HTTP_CODE);
  $body = substr($resp, $header_size);
  $raw_headers = substr($resp, 0, $header_size);
  curl_close($ch);
  return [$body, $raw_headers, $status, null];
}

// Resolve relative URI against base
function resolve_url($base, $rel){
  // If already absolute (http/https/data), return as is
  if (preg_match('#^(?:[a-z]+:)?//#i', $rel)) return $rel;
  // Remove query/fragment from base path
  $p = parse_url($base);
  $scheme = $p['scheme'] ?? 'http';
  $host = $p['host'] ?? '';
  $port = isset($p['port']) ? (':'.$p['port']) : '';
  $path = $p['path'] ?? '/';
  if (substr($rel,0,1) === '/') {
    $new_path = $rel;
  } else {
    $dir = preg_replace('#/[^/]*$#', '/', $path);
    $new_path = $dir . $rel;
  }
  // Normalize "../" and "./"
  $segments = [];
  foreach (explode('/', $new_path) as $seg) {
    if ($seg === '' || $seg === '.') continue;
    if ($seg === '..') { array_pop($segments); continue; }
    $segments[] = $seg;
  }
  $norm = '/'.implode('/', $segments);
  return $scheme.'://'.$host.$port.$norm;
}

// Detect extension to guess type
$ext = strtolower(pathinfo(parse_url($u, PHP_URL_PATH) ?? '', PATHINFO_EXTENSION));

list($body, $raw_headers, $code, $err) = curl_fetch($u, $headers);
if ($body === null) { http_response_code(502); echo "upstream error: $err"; exit; }
if ($code < 200 || $code >= 400) { http_response_code($code); echo "upstream status $code"; exit; }

// Heuristics to detect m3u8
$is_m3u8 = ($ext === 'm3u8') || (strpos($body, '#EXTM3U') !== false);

// Forward Range/length only for segments
if (!$is_m3u8) {
  // Best-effort content-type
  if (in_array($ext, ['ts','m2ts','mp2t'])) header('Content-Type: video/MP2T');
  elseif (in_array($ext, ['m4s'])) header('Content-Type: application/octet-stream');
  elseif (in_array($ext, ['mp4'])) header('Content-Type: video/mp4');
  elseif (in_array($ext, ['aac'])) header('Content-Type: audio/aac');
  else header('Content-Type: application/octet-stream');

  // Try to forward partial content status if upstream provided Content-Range
  if (preg_match('/^Content-Range:\s*(.+)$/mi', $raw_headers, $m)) {
    header('Content-Range: '.$m[1]);
    http_response_code(206);
  }
  if (preg_match('/^Accept-Ranges:\s*(.+)$/mi', $raw_headers, $m)) {
    header('Accept-Ranges: '.$m[1]);
  } else {
    header('Accept-Ranges: bytes');
  }
  echo $body;
  exit;
}

// Playlist rewriting
header('Content-Type: application/vnd.apple.mpegURL; charset=utf-8');
$lines = preg_split("/\r?\n/", $body);
$out = [];
$pendingStreamInf = false;

foreach ($lines as $line) {
  $trim = trim($line);
  if ($trim === '' || $trim[0] === '#') {
    $out[] = $line;
    // When line is #EXT-X-STREAM-INF, the next non-comment line is a variant URI we must rewrite
    if (stripos($trim, '#EXT-X-STREAM-INF') === 0) $pendingStreamInf = true;
    continue;
  }
  // URL/Path line (variant or segment)
  $target = $trim;
  if ($pendingStreamInf) { $pendingStreamInf = false; }
  // Resolve relative against base playlist URL
  $abs = resolve_url($u, $target);
  // Wrap through proxy (always)
  $prox = 'hls_proxy.php?u=' . urlencode(base64_encode($abs));
  $out[] = $prox;
}

echo implode("\n", $out);
?>