<?php
require_once __DIR__ . "/config.php";
allow_json();

header("Access-Control-Allow-Origin: *");
header("Access-Control-Allow-Methods: GET, POST, OPTIONS");
header("Access-Control-Allow-Headers: Content-Type");
header("Cache-Control: no-store, no-cache, must-revalidate, max-age=0");
if (($_SERVER['REQUEST_METHOD'] ?? '') === 'OPTIONS') { exit; }

define("CHAT_FILE", DATA_DIR . "/chat.json");
define("USERS_FILE", DATA_DIR . "/chat_users.json"); // uid => name

// Asegurar ficheros
if (!file_exists(CHAT_FILE)) {
  @file_put_contents(CHAT_FILE, json_encode(["messages"=>[]], JSON_PRETTY_PRINT|JSON_UNESCAPED_UNICODE));
}
if (!file_exists(USERS_FILE)) {
  @file_put_contents(USERS_FILE, json_encode(["users"=>[]], JSON_PRETTY_PRINT|JSON_UNESCAPED_UNICODE));
}

function load_json_file($path, $rootKey){
  $raw = @file_get_contents($path);
  $db = $raw ? json_decode($raw, true) : null;
  if (!is_array($db) || !isset($db[$rootKey]) || !is_array($db[$rootKey])) $db = [$rootKey=>[]];
  return $db;
}
function save_json_file($path, $db){
  $fp = @fopen($path, "c+");
  if (!$fp) { http_response_code(500); echo json_encode(["error"=>"OPEN_FAIL"]); exit; }
  if (!flock($fp, LOCK_EX)) { fclose($fp); http_response_code(500); echo json_encode(["error"=>"LOCK_FAIL"]); exit; }
  ftruncate($fp, 0);
  rewind($fp);
  $ok = fwrite($fp, json_encode($db, JSON_PRETTY_PRINT|JSON_UNESCAPED_UNICODE));
  fflush($fp); flock($fp, LOCK_UN); fclose($fp);
  if ($ok === false) { http_response_code(500); echo json_encode(["error"=>"WRITE_FAIL"]); exit; }
}

// Helpers
function load_chat(){ return load_json_file(CHAT_FILE, "messages"); }
function save_chat($db){ return save_json_file(CHAT_FILE, $db); }
function load_users(){ return load_json_file(USERS_FILE, "users"); }
function save_users($db){ return save_json_file(USERS_FILE, $db); }

$method = $_SERVER['REQUEST_METHOD'] ?? 'GET';

if ($method === 'GET') {
  // /api/chat.php?since=ID (opcional)
  // /api/chat.php?uid=...&me=1   → devuelve el nombre fijado de ese uid (si existe)
  if (isset($_GET['me']) && ($_GET['me'] === '1' || $_GET['me'] === 'true')) {
    $uid = trim($_GET['uid'] ?? '');
    $users = load_users();
    $name = '';
    if ($uid !== '' && isset($users['users'][$uid])) {
      $name = $users['users'][$uid];
    }
    echo json_encode(["ok"=>true, "name"=>$name], JSON_UNESCAPED_UNICODE);
    exit;
  }

  $since = trim($_GET['since'] ?? '');
  $db = load_chat();
  if ($since !== '') {
    $pos = -1;
    foreach ($db['messages'] as $i=>$m) {
      if ((string)($m['id'] ?? '') === $since) { $pos = $i; break; }
    }
    $msgs = ($pos >= 0) ? array_slice($db['messages'], $pos+1) : $db['messages'];
  } else {
    // últimas 100
    $msgs = array_slice($db['messages'], -100);
  }
  echo json_encode(["ok"=>true, "messages"=>$msgs], JSON_UNESCAPED_UNICODE);
  exit;
}

if ($method === 'POST') {
  $raw = file_get_contents('php://input') ?: '';
  $ct  = $_SERVER['CONTENT_TYPE'] ?? '';
  if (stripos($ct, 'application/json') !== false) {
    $b = json_decode($raw, true);
  } else {
    parse_str($raw, $form); $b = $form;
  }

  $uid  = trim((string)($b['uid'] ?? ''));
  $name = trim((string)($b['name'] ?? ''));
  $text = trim((string)($b['text'] ?? ''));

  // Fallback de uid (no recomendado, pero por si el cliente no lo envía)
  if ($uid === '') {
    $uid = 'ip:' . ($_SERVER['REMOTE_ADDR'] ?? '0.0.0.0');
  }

  if ($text === '') {
    http_response_code(400); echo json_encode(["ok"=>false,"error"=>"BAD_INPUT"]); exit;
  }

  // límites
  $name = mb_substr($name, 0, 24);
  $text = mb_substr($text, 0, 500);

  // Cargar usuarios y fijar nombre por uid
  $users = load_users();
  if (!isset($users['users'][$uid])) {
    if ($name === '') { http_response_code(400); echo json_encode(["ok"=>false,"error"=>"NAME_REQUIRED"]); exit; }
    $users['users'][$uid] = $name;               // Primera vez → guardamos nombre
    save_users($users);
  }
  // Para mensajes, usamos SIEMPRE el nombre fijado del uid
  $fixedName = $users['users'][$uid];

  // Anti-spam simple: 1 msg / 2s por IP
  $ip = $_SERVER['REMOTE_ADDR'] ?? '0.0.0.0';
  $now = time();

  $db = load_chat();
  if (!empty($db['messages'])) {
    for ($i=count($db['messages'])-1; $i>=0 && $i>=count($db['messages'])-10; $i--) {
      $m = $db['messages'][$i];
      if (($m['ip'] ?? '') === $ip) {
        if ($now - (intval($m['ts'] ?? 0)) < 2) {
          http_response_code(429); echo json_encode(["ok"=>false,"error"=>"RATE_LIMIT"]); exit;
        }
        break;
      }
    }
  }

  $msg = [
    "id"  => uniqid(),
    "ts"  => $now,
    "uid" => $uid,
    "name"=> htmlspecialchars($fixedName, ENT_QUOTES|ENT_SUBSTITUTE, 'UTF-8'),
    "text"=> htmlspecialchars($text, ENT_QUOTES|ENT_SUBSTITUTE, 'UTF-8'),
    "ip"  => $ip
  ];
  $db['messages'][] = $msg;

  // Mantener máx 300
  if (count($db['messages']) > 300) $db['messages'] = array_slice($db['messages'], -300);

  save_chat($db);
  echo json_encode(["ok"=>true, "message"=>$msg], JSON_UNESCAPED_UNICODE);
  exit;
}

http_response_code(405); echo json_encode(["ok"=>false,"error"=>"METHOD_NOT_ALLOWED"]);